package loquebot.body;

import java.util.ArrayList;

import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.MessageType;
import cz.cuni.pogamut.MessageObjects.Triple;

import cz.cuni.pogamut.MessageObjects.Item;
import cz.cuni.pogamut.MessageObjects.Player;
import cz.cuni.pogamut.MessageObjects.Spawn;

import loquebot.Main;
import loquebot.util.LoqueListener;
import loquebot.memory.LoqueMemory;

/**
 * Responsible for handling traveling througout the map.
 *
 * <p>Main purpose of this class is to provide simple (but powerful) API for
 * safe traveling througout the map. Whether it is just a simple relocation on
 * short distances; or a complicated journey along navigation points.</p>
 *
 * <p>Travel tickets are issued in order to prevent drives to fight over the
 * navigation without knowing about each other. Such alternating could lead to
 * confusion anyway. The drives are therefore automatically forced to create
 * new travel tickets each time another drive uses the travel API.</p>
 *
 * <p>When a drive wants to travel to a specific location, it creates a travel
 * ticket through one of the <i>init*()</i> methods and then keeps calling
 * {@link #keepTraveling} method with this ticket to pass control to navigation
 * logic in this API.<p>
 *
 * @author Juraj Simlovic [jsimlo@matfyz.cz]
 * @version Tested on Pogamut 2 platform version 1.0.5.
 */
public class LoqueTravel extends LoqueTravelBase
{
    /**
     * Initializes new travel ticket to all of the items from the given list
     * of items.
     *
     * <p>The items are visited in progressive sequence, one by one, based on
     * how far they are from the agent and from each other.</p>
     *
     * <p>Note: If the agent fails to visit some of the items due to travel
     * troubles (e.g. unable to reach some locations), he might want to decide
     * to try to re-visit them later (e.g. trying from different transit camps).
     * Nonetheless, all of the items will either get visited or forsaken.</p>
     *
     * @param items List of items to travel to.
     * @return Id of the travel ticket. Use this id in {@link #keepTraveling}.
     * Only positive values are valid tickets. Negative values mean error.
     */
    public int initTicketToAll (ArrayList<Item> items)
    {
        // init travel ticket
        return initTicket (items, true);
    }

    /**
     * Initializes new travel ticket to the nearest item from the given list of
     * items.
     *
     * @param items List of items from which to choose.
     * @return Id of the travel ticket. Use this id in {@link #keepTraveling}.
     * Only positive values are valid tickets. Negative values mean error.
     */
    public int initTicketToNearest (ArrayList<Item> items)
    {
        // init travel ticket
        return initTicket (items, false);
    }

    /**
     * Initializes new travel ticket to one random item from the given list of
     * items.
     *
     * @param items List of items from which to choose.
     * @return Id of the travel ticket. Use this id in {@link #keepTraveling}.
     * Only positive values are valid tickets. Negative values mean error.
     */
    public int initTicketToRandom (ArrayList<Item> items)
    {
        ArrayList<Item> chosen = new ArrayList<Item> (1);

        // is there an item to pick?
        if (items.size () > 0)
        {
            int pick = (int) (Math.random () * items.size ());
            // pick one random item
            chosen.add(items.get (pick));
        }

        // init travel ticket
        return initTicket (chosen, false);
    }

    /**
     * Initializes new travel ticket to the destination of the given player.
     *
     * @param player Player to which location to run to.
     * @param timeout Maximum amount of time to travel. Use -1 to for unlimited.
     * @return Id of the travel ticket. Use this id in {@link #keepTraveling}.
     * Only positive values are valid tickets. Negative values mean error.
     */
    public int initTicketToPlayer (Player player, int timeout)
    {
        // is the player reachable?
        if (player.reachable)
        {
            // init travel ticket to directly reachable target
            return initTicket (player.location, new ArrayList<Triple> (0), false, timeout);
        }
        else
        {
            ArrayList<Triple> list = new ArrayList<Triple> (1);

            // use this one player location as destination
            list.add (player.location);

            // init travel ticket to directly unreachable targets
            return initTicket (null, list, false, timeout);
        }
    }

    /*========================================================================*/

    /**
     * Continues traveling with the given travel ticket.
     *
     * @param ticket Id of the travel ticket.
     * @return True, if still traveling. False otherwise.
     */
    public boolean keepTraveling (int ticket)
    {
        // if the ticket matches
        if (ticketId == ticket)
        {
            // continue the travel
            if (!manager.keepTraveling ())
            {
                //main.stopAgentSoft ();
                //throw new RuntimeException ("Done!");
                return false;
            }
            return true;
        }

        // no, we're not traveling anymore
        log.finest ("Travel.keepTraveling(): failed: obsolete ticket");
        return false;
    }

    /*========================================================================*/

    /**
     * Kills any ongoing travel ticket, which in return forces all drives
     * to make new travel tickets, if they want to. This practice is used
     * after respawns, when the agent is relocated by the engine; or during
     * combats, when the agent comes down from the recent traveling paths.
     */
    public void killTicket ()
    {
        // kill whatever the ticket by issuing a new one
        ticketId++;
    }

    /*========================================================================*/

    /**
     * Listening class for messages from engine.
     */
    private class Listener extends LoqueListener
    {
        /**
         * Agent just spawned into the game.
         * @param msg Message to handle.
         */
        private void msgSpawn (Spawn msg)
        {
            // reset pursue info
            killTicket ();
        }

        /**
         * Message switch.
         * @param msg Message to handle.
         */
        protected void processMessage (MessageObject msg)
        {
            switch (msg.type)
            {
                case SPAWN:
                    msgSpawn ((Spawn) msg);
                    return;
            }
        }

        /**
         * Constructor: Signs up for listening.
         */
        private Listener ()
        {
            body.addTypedRcvMsgListener(this, MessageType.SPAWN);
        }
    }

    /** Listener. */
    private LoqueListener listener;

    /*========================================================================*/

    /**
     * Constructor.
     * @param main Agent's main.
     * @param memory Loque memory.
     */
    public LoqueTravel (Main main, LoqueMemory memory)
    {
        super (main, memory);
        // initialize listening
        new Listener ();
    }
}